## Jim Stevens
## 13/11/20 UPDATED 4/6/24
## Version 4.3.3
## Code module to read in data from summary file
## for sq wave 400 par, 18h giving A gs and Fq' / Fm' info

## Housekeeping #### 

rm(list=ls()) # Clear R workspace 

library(openxlsx) # open excel files
library(compiler) # rapid calculations - not really necessary here
library(tidyverse) # for all the data wrangling functions
library(cowplot) # combining plots (ggplot can do this a bit) 

enableJIT(2) # Runs compiled code rapidly (level 2 here - small loops)

## Read in multiple files ####

d <- NULL 

for (f in dir(".//Data/Constant Light Test", pattern=".xlsx", recursive=T)) {
  # this function finds the name of each .csv file in turn
  if(strsplit(f,split=0)[[1]][1]!=2) next # only include 400 PAR reps
  print(f) # This allows a quick check that we are reading in all the files
  
  tmp <- read.xlsx(paste(".//Data/Constant Light Test/",f,sep=""),
                   sheet="Measurements",
                   rows=c(17,19:250)
                   )
  # the file 'tmp' now holds the CURRENT file specified by the variable, f
  # we only select the first 111 rows
  
  # now we parse the information in the variable, f
  Ind <- strsplit(f, "_")[[1]] # splits up f into elements separated by "_"
  Date <- paste(strsplit(Ind[1],split="-")[[1]][1],
                strsplit(Ind[1],split="-")[[1]][2],
                strsplit(Ind[1],split="-")[[1]][3],
                sep="")
  Plant_Number <- strsplit(Ind[1],"-")[[1]][4] 
  # Returns the rep number / individual plant identifier

  ## Recombine the data
  d <- rbind(d,data.frame(ID=f,
                          Date=Date, 
                          Plant_Number=Plant_Number,  
  # This function takes the object d and turns it into a dataframe. 
  # With each new file identified by f, we parse the filename and 
  # use the relevant data then look inside each .csv file and 
  # extract the variables we want.
                          Obs=tmp$obs,
                          Time=tmp$elapsed,
                          A=tmp$A, 
                          gs=tmp$gsw,
                          FvpFmp=tmp$FvpFmp,
                          FqpFvp=tmp$PhiPS2/tmp$FvpFmp,
                          FqpFmp=tmp$PhiPS2,
                          NPQ=tmp$NPQ,
                          Qin=tmp$Qin
  ))
  # at the end of each loop of the function, we add (rbind) the next .csv
  # file's data to the bottom of the dataframe d in a row-wise manner 
}

write.xlsx(file=".//Data/Constant Light Test/Combined 18h Diurnal Data 400 PAR.xlsx",d)

# Tidy up the structure
str(d)
d$ID <- as.factor(d$ID)
d$Date <- as.Date(d$Date,format='%y-%m-%d')
#d$Plant_Number <- as.integer(d$Plant_Number)
d$Amax <- max(d$A)
str(d)

d1 <- d %>% # get last value before lights on
  group_by(ID) %>%
  filter(Time<4800 & Qin<10) %>%
  arrange(Time) %>%
  filter(row_number()==max(row_number()))

d2 <- d %>% # Get the last time point after lights off
  group_by(ID) %>%
  filter(Time>4800 & Qin<10) %>%
  arrange(Time) %>%
  filter(row_number()==1)

d3 <- d %>% # Get all points while lights on
  filter(Qin>10)

d <- rbind(d1,d2,d3)

## Create some figures from the data ####

for (g in unique(d$ID)) {
  # We use the for function to search our dataframe, d

    gg <- d %>%
      filter(ID==g) # create a new dataframe as a subset of d for this rep
  
    model <- gg %>%
      summarise(Mean=mean(A))
    
# create a plot of the data + model for this rep  
  gg_fig <- ggplot(gg,aes(x=Obs,y=A)) + 
    # The first line creates the ggplot object
    # We take the data from the dataframe gg
    # The aes command gives the aesthetics; here x and y values
    # But we can also colour the lines here is we were plotting > 1
    # data series at a time
    geom_line(linewidth=2)+
    # the syntax 'geom_line' tells R to create a line plot
    labs(title=g) +
    # We add a title at the top of the figure using the 'labs' command
    geom_hline(yintercept=model$Mean,colour="red",linewidth=2,linetype="dashed")
    # We add a horizontal line based on our model using geom_hline

  print(gg_fig) # optional - quick check on the plot window bottom right

  save_plot(file=paste(".//Outputs/Constant Light Tests/",g,".png",sep=""),
          gg_fig,base_width=8)
  # save the plots one at a time so we can inspect each data series

} # Note that for loops must start and end with curly brackets

## Combine the reps into a single time series plot ####

dA <- d %>%
  group_by(Obs) %>%
  dplyr::summarise(N=n(),Mean=mean(A),SE=sd(A)/sqrt(N),Time=Time[1],.groups="keep")
dA$Max <- max(dA$Mean)
dA <- dA[-c(201),] # one that slipped through
dA$Time_Elapsed <- round(((dA$Time-3000)/3600),3)
dA$Tlead <- lead(dA$Time,default=69001)
dA$Tdiff <- dA$Tlead-dA$Time

dA1 <- dA[-c(1,200),]
A.Potential <- sum((dA$Max*dA$Tdiff)/max(dA$Time)) 
A.Actual <- sum((dA$Mean*dA$Tdiff)/max(dA$Time))
A.Delta <- round((1-A.Actual/A.Potential)*100,1)
A.D <- paste0(A.Delta,"% 'missing' Assimilation")
A.D
assim <- data.frame(c("Potential_A","Actual_A","Delta_A_pc"),
                    c(A.Potential,A.Actual,A.Delta))
names(assim) <- c("Measurement","Value")                       
capture.output(assim,file="./Outputs/Final Plots/Stats/Initial_A_test_Data.txt")

A_plot <- ggplot(dA,aes(x=Time_Elapsed,ymin=Mean-SE,ymax=Mean+SE))+
  geom_ribbon(mapping=aes(y=dA$Mean,ymax=dA$Max),alpha=0.2,fill="goldenrod1")+
  geom_point(aes(y=Mean,colour="A"),size=1)+
  geom_ribbon(alpha=0.2) +
  theme_bw() +
  theme(panel.grid=element_blank(),
        axis.title=element_text(size=11)) +
  ylab(expression(paste(Assimilation," (",~mu,mol,~CO[2],~m^-2,~s^-1," )")))+
  xlab("Time (h)") +
  scale_x_continuous(breaks=c(0,3,6,9,12,15,18))+
  scale_y_continuous(limits=c(-2,16),breaks=c(0,5,10,15,20))+
  scale_colour_manual(name="",values=c("A"="orange"))+
  xlim(0,18)+
  annotate("text",x=11,y=12,
           label=A.D,
           hjust=0,size=3)
plot(A_plot)

save_plot(file=".//Outputs/Constant Light Tests/Overall_Assimilation.png",A_plot,base_width=8)

dgs <- d %>%
  group_by(Obs) %>%
  summarise(N=n(),Mean=mean(gs),SE=sd(gs)/sqrt(N),.groups="keep") %>%
  filter(Obs<110)
dgs$Time_Elapsed <- (dgs$Obs*10-10)/60


gs_plot <- ggplot(dgs,aes(x=Time_Elapsed,ymin=Mean-SE,ymax=Mean+SE))+
  geom_point(aes(y=Mean,colour="Gs"),size=1)+
  geom_ribbon(alpha=0.2) +
  theme_bw() +
  theme(panel.grid=element_blank(),
        axis.title=element_text(size=9)) +
  ylab(expression(paste(italic(g)[s]," ( ",~mol,~m^-2,~s^-1," )")))+
  xlab("ZT Time(hrs)") +
  scale_x_continuous(breaks=c(0,3,6,9,12,15,18))+
  scale_colour_manual(name="",values = c("Gs"="purple"))

plot(gs_plot)

save_plot(file=".//Outputs/Constant Light Tests/Overall_Stomatal_Conductance.png",gs_plot,base_width=8)
save_plot(file=".//Outputs/Final Plots/Supplementary Information/S1_Constant_Light_400_Stomatal_Conductance.png",gs_plot,base_width=8)

## Fluorescence Analysis ####

# Plot all
fluo <- NULL
fluo <- d %>%
  group_by(Obs) %>%
  summarise(N_count=n(),FvpFmp_m=mean(FvpFmp),FvpFmp_se=sd(FvpFmp)/sqrt(N_count),
            N_count=n(),FqpFvp_m=mean(FqpFvp),FqpFvp_se=sd(FqpFvp)/sqrt(N_count),
            N_count=n(),FqpFmp_m=mean(FqpFmp),FqpFmp_se=sd(FqpFmp)/sqrt(N_count),
            N_count=n(),NPQ_m=mean(NPQ),NPQ_se=sd(NPQ)/sqrt(N_count),
            .groups="drop") #%>%
fluo <- fluo[-c(201),] 
fluo$Time <- dA$Time
fluo <- fluo[-c(1,200),]
fluo$Time_Elapsed <- round(((fluo$Time-3000)/3600),3)
fluo$Tlead <- lead(fluo$Time,default=68103)
fluo$Tdiff <- fluo$Tlead-fluo$Time
fluo <- fluo %>%
  pivot_longer(cols=!Time_Elapsed,
               names_to = c("Measurement","Stat"),
               names_sep = "_",
               values_to = "Value") %>%
  pivot_wider(id_cols=Time_Elapsed,
              names_from = c("Measurement","Stat"), 
              values_from = "Value",
              values_fn = mean) 
#fluo
 # fluo$Time_Elapsed <- ((fluo$Obs-3)/106)*18 # rescale to 18 hour day
  
val1 <- paste("'Fq'/Fm'","Fq'/Fv'","Fv'/Fm'")

fluo_plot_1 <- ggplot(fluo,aes(x=Time_Elapsed))+
  geom_point(aes(y=FvpFmp_m,colour="FvpFmp"))+
  geom_ribbon(aes(ymin=FvpFmp_m-FvpFmp_se,
                    ymax=FvpFmp_m+FvpFmp_se)
              ,alpha=0.2)+
  geom_point(aes(y=FqpFmp_m,colour="FqpFmp")) +
  geom_ribbon(aes(ymin=FqpFmp_m-FqpFmp_se,
                ymax=FqpFmp_m+FqpFmp_se),
              alpha=0.2) +
  geom_point(aes(y=FqpFvp_m,colour="FqpFvp")) +
  geom_ribbon(aes(ymin=FqpFvp_m-FqpFvp_se,
                    ymax=FqpFvp_m+FqpFvp_se),
              alpha=0.2) +
  theme_bw() + 
  theme(panel.grid=element_blank(),
        axis.title=element_text(size=11),
        legend.text=element_text(size=11)) +
  ylab("Chlorophyll \nFluorescence (a.u.)")+
  xlab("Time (h)")+
  scale_colour_manual(name="",values = c("FqpFmp"="red",
                                         "FqpFvp"="limegreen",
                                         "FvpFmp"="darkblue"),
                      labels=c("Fq'/Fm'","Fq'/Fv'","Fv'/Fm'"))+
  scale_x_continuous(breaks=c(0,3,6,9,12,15,18))+
  guides(fill="none")

fluo.leg <- get_legend(fluo_plot_1+theme(legend.direction="horizontal"))
fluo_plot_1
ggsave(filename="./Outputs/Constant Light Tests/Fluoresence.png",fluo_plot_1)

fluo_plot_2 <- ggplot(fluo,aes(x=Time_Elapsed,
                               ))+
  geom_point(aes(y=NPQ_m,colour="NPQ")) +
  geom_ribbon(aes(ymin=NPQ_m-NPQ_se, ymax=NPQ_m+NPQ_se),alpha=0.2)+
  theme_bw() + 
  theme(panel.grid=element_blank(),
        axis.title=element_text(size=11)) +
  ylab("Non-Photochemical\nQuenching")+
  xlab("Time (h)")+
  scale_colour_manual(name="",values = c("NPQ"="darkgreen"))+
  scale_fill_manual(name="",values=c("NPQ"="darkgreen"))+
  scale_x_continuous(breaks=c(0,3,6,9,12,15,18)) +
  guides(fill="none")  

fluo_plot_2
ggsave("./Outputs/Constant Light Tests/NPQ.png",fluo_plot_2)
ggsave("./Outputs/Final Plots/Supplementary Information/NPQ.png",fluo_plot_2)

#### NEW COMBINED PLOTS ####

## Combined plots ####
# This section uses the cowplot functionality
# Similar functionality can be found in ggplot or other packages

combined <- plot_grid(A_plot + theme(axis.text.x=element_blank(),
                                     axis.title.x = element_blank(),
                                     legend.position="none"),
                      fluo_plot_1+theme(legend.position="none"),
                      nrow=2,
                      rel_heights=c(0.8,1),
                      align="v",
                      labels="AUTO")

combined # check the output looks about right
combined1 <- plot_grid(combined,fluo.leg,nrow=2,rel_heights = c(1,0.1))
combined1
save_plot(file=".//Outputs/Constant Light Tests/Combined_A_Chl_f.png",
          combined,base_height=10)
save_plot("./Outputs/Final Plots/Fig_1_A_ChlF_diurnals.png",
          combined1,base_height=9,base_width=7)

# Note that your saved file might look a bit different to your print output above
# particularly depending on the base_height or base_width you pick
# so it's best to check them in the folder where you saved them

# Combine fluorescence figures

leg1 <- get_legend(A_plot+theme(legend.direction = "horizontal",
                               legend.justification="left" ,
                               legend.box.just = "bottom"))
leg2 <- get_legend(gs_plot+theme(legend.direction = "horizontal",
                                 legend.justification="left" ,
                                 legend.box.just = "bottom"))
leg3 <- get_legend(fluo_plot_1+theme(legend.direction = "horizontal",
                                     legend.justification="center" ,
                                     legend.box.just = "bottom"))
leg4 <- get_legend(fluo_plot_2+theme(legend.direction = "horizontal",
                                     legend.justification="center" ,
                                     legend.box.just = "bottom"))


ags_leg <- plot_grid(leg1,leg2,align="h",nrow=1)
ags_leg
ags_combined <- plot_grid(A_plot + theme(axis.text.x=element_blank(),
                                         axis.title.x = element_blank(),
                                         legend.position="none"),
                          gs_plot+theme(legend.position="none"),
                          ags_leg,
                          nrow=3,
                          rel_heights=c(0.85,1,0.1),
                          align="v",
                          labels=c("A",NULL,"B"),
                          label_size=10)

ags_combined

fluo_leg <- plot_grid(leg3,NULL,leg4,align="h",nrow=1)
fluo_leg

fluo_combined_1 <- plot_grid(fluo_plot_1 + theme(axis.text.x=element_blank(),
                                               axis.title.x = element_blank(),
                                               legend.position="none"),
                      fluo_plot_2 + theme(legend.position="none"),
                      fluo_leg,
                      nrow=3,
                      rel_heights=c(0.85,1,0.1),
                      align="v",
                      labels=c("C",NULL,"D"),
                      label_size=10)
fluo_combined_1


fluo_combined_2 <- plot_grid(ags_combined,fluo_combined_1,
                             nrow=1,
                             align="h")
fluo_combined_2

#save_plot("./Outputs/A_gs_ChlF_diurnals.png",fluo_combined_2)

